<?php
if (!defined('ABSPATH')) exit;

// Adiciona o gancho para o evento agendado pelo WP Cron
add_action('limber_sync_precos_event', 'limber_sync_precos');

/**
 * Função principal que busca e salva os preços da API.
 */
function limber_sync_precos() {
    $options = get_option('limber_settings');
    $usuario = $options['limber_usuario'] ?? '';
    $senha = $options['limber_senha'] ?? '';

    if (empty($usuario) || empty($senha)) {
        update_option('limber_last_sync_status', 'Falha: Usuário ou senha da API não configurados.');
        return;
    }

    // Obter Token
    $response_auth = wp_remote_post('https://card.limbersoftware.com.br/api/auth', [
        'headers' => ['Content-Type' => 'application/json'],
        'body' => json_encode(['usuario' => $usuario, 'senha' => $senha]),
        'timeout' => 20
    ]);

    if (is_wp_error($response_auth) || wp_remote_retrieve_response_code($response_auth) !== 200) {
        update_option('limber_last_sync_status', 'Falha na autenticação com a API.');
        return;
    }

    $token = json_decode(wp_remote_retrieve_body($response_auth), true)['token'] ?? null;
    if (!$token) {
        update_option('limber_last_sync_status', 'Falha: Token não recebido da API.');
        return;
    }
    
    // Preparar parâmetros base
    $params = [
        'sku'        => (int) ($options['limber_sku'] ?? 0),
        'idparceiro' => (int) ($options['limber_id_parceiro'] ?? 0),
        'receita'    => (int) ($options['limber_receita'] ?? 0),
        'meiovenda'  => 'WEB',
    ];

    // ETAPA 1: Buscar todos os dias com disponibilidade nos próximos 90 dias.
    $data_inicio = date('Y-m-d');
    $data_fim = date('Y-m-d', strtotime('+90 days'));
    
    $disp_response = wp_remote_post('https://card.limbersoftware.com.br/api/cross/consulta/disponibilidade/calendario', [
        'headers' => ['Authorization' => 'Bearer ' . $token, 'Content-Type'  => 'application/json'],
        'body'    => json_encode(array_merge($params, ['datainicial' => $data_inicio.'T00:00:00', 'datafinal' => $data_fim.'T23:59:59'])),
        'timeout' => 45,
    ]);
    
    if (is_wp_error($disp_response)) {
        update_option('limber_last_sync_status', 'Falha: Erro de conexão ao buscar disponibilidade.');
        return;
    }

    $dias_disponiveis = json_decode(wp_remote_retrieve_body($disp_response), true);
    if (!is_array($dias_disponiveis) || empty($dias_disponiveis)) {
        update_option('limber_last_sync_status', 'Sucesso: Nenhum dia com disponibilidade encontrado nos próximos 90 dias.');
        update_option('limber_last_sync_time', time());
        return;
    }

    global $wpdb;
    $dias_processados = 0;
    $erros_encontrados = [];

    // ETAPA 2: FAZER UM LOOP EM CADA DIA DISPONÍVEL E BUSCAR O PREÇO INDIVIDUALMENTE.
    foreach ($dias_disponiveis as $dia) {
        if (!isset($dia['disponivel']) || $dia['disponivel'] <= 0) {
            continue;
        }

        $data_key = substr($dia['data'], 0, 10);

        // Busca o preço para este único dia. Usamos a sugestão de "data + 1 dia" para dataFim, que se provou funcional.
        $data_fim_loop = date('Y-m-d', strtotime($data_key . ' +1 day'));

        $preco_response = wp_remote_post('https://card.limbersoftware.com.br/api/cross/consulta/configpreco', [
            'headers' => ['Authorization' => 'Bearer ' . $token, 'Content-Type'  => 'application/json'],
            'body'    => json_encode(array_merge($params, ['data' => $data_key.'T00:00:00', 'dataFim' => $data_fim_loop.'T00:00:00'])),
            'timeout' => 15,
        ]);
        
        if (is_wp_error($preco_response) || wp_remote_retrieve_response_code($preco_response) !== 200) {
            $erros_encontrados[] = $data_key;
            continue;
        }

        $configs_preco = json_decode(wp_remote_retrieve_body($preco_response), true);
        if (!is_array($configs_preco) || isset($configs_preco['message']) || empty($configs_preco)) {
            $erros_encontrados[] = $data_key;
            continue;
        }

        // Como a resposta pode conter mais de um dia, garantimos que estamos pegando o dia correto.
        $config_do_dia = null;
        foreach($configs_preco as $c) {
            if (substr($c['data'], 0, 10) === $data_key) {
                $config_do_dia = $c;
                break;
            }
        }
        if (!$config_do_dia) continue;
        
        $categorias = $config_do_dia['configPreco']['categorias'] ?? [];
        if (empty($categorias)) continue;

        $menor_preco = PHP_FLOAT_MAX;
        foreach ($categorias as $categoria) {
            if (isset($categoria['valorUnitario']) && is_numeric($categoria['valorUnitario']) && $categoria['valorUnitario'] < $menor_preco) {
                $menor_preco = (float) $categoria['valorUnitario'];
            }
        }

        if ($menor_preco !== PHP_FLOAT_MAX) {
            $wpdb->replace(LIMBER_CALENDAR_TABLE,
                ['data_visita' => $data_key, 'menor_preco' => $menor_preco],
                ['%s', '%f']
            );
            $dias_processados++;
        }
        
        sleep(1);
    }

    $status_final = "Sucesso: $dias_processados dias com preços foram atualizados/inseridos.";
    if (!empty($erros_encontrados)) {
        $status_final .= " Falha ao buscar preço para os dias: " . implode(', ', $erros_encontrados) . ".";
    }

    update_option('limber_last_sync_time', time());
    update_option('limber_last_sync_status', $status_final);
}